<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\DeviceToken;
use App\Models\Package;
use App\Models\Referral;
use App\Models\SignalHistory;
use App\Models\Transaction;
use App\Models\AdminNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function home()
    {
        // AdminNotification
        $pageTitle    = 'Dashboard';
        $user         = auth()->user();
        $totalTrx     = Transaction::where('user_id', $user->id)->count();
        $totalSignal  = SignalHistory::where('user_id', $user->id)->count();
        $latestTrx    = Transaction::where('user_id', $user->id)->orderBy('id', 'DESC')->limit(10)->get();
        $totalDeposit = Deposit::where('user_id', $user->id)->where('status', Status::PAYMENT_SUCCESS)->sum('amount');
        return view('Template::user.dashboard', compact('pageTitle', 'user', 'totalDeposit', 'totalTrx', 'latestTrx', 'totalSignal'));
    }

    public function depositHistory(Request $request)
    {
        $pageTitle = 'Deposit History';
        $deposits  = auth()->user()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.deposit_history', compact('pageTitle', 'deposits'));
    }


    public function transactions()
    {
        $pageTitle    = 'Transactions';
        $remarks      = Transaction::distinct('remark')->orderBy('remark')->get('remark');
        $transactions = Transaction::where('user_id', auth()->id())->searchable(['trx'])->filter(['trx_type', 'remark'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.transactions', compact('pageTitle', 'transactions', 'remarks'));
    }




    public function userData()
    {
        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $pageTitle  = 'User Data';
        $info       = json_decode(json_encode(getIpInfo()), true);
        $mobileCode = @implode(',', $info['code']);
        $countries  = json_decode(file_get_contents(resource_path('views/partials/country.json')));

        return view('Template::user.user_data', compact('pageTitle', 'user', 'countries', 'mobileCode'));
    }

    public function userDataSubmit(Request $request)
    {

        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $countryData  = (array)json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $request->validate([
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
            'username'     => 'required|unique:users|min:6',
            'mobile'       => ['required', 'regex:/^([0-9]*)$/', Rule::unique('users')->where('dial_code', $request->mobile_code)],
        ]);


        if (preg_match("/[^a-z0-9_]/", trim($request->username))) {
            $notify[] = ['info', 'Username can contain only small letters, numbers and underscore.'];
            $notify[] = ['error', 'No special character, space or capital letters in username.'];
            return back()->withNotify($notify)->withInput($request->all());
        }

        $user->country_code = $request->country_code;
        $user->mobile       = $request->mobile;
        $user->username     = $request->username;


        $user->address           = $request->address;
        $user->city              = $request->city;
        $user->state             = $request->state;
        $user->zip               = $request->zip;
        $user->country_name      = @$request->country;
        $user->dial_code         = $request->mobile_code;
        $user->telegram_username = $request->telegram_username;
        $user->profile_complete  = Status::YES;
        $user->save();

        return to_route('user.home');
    }


    public function addDeviceToken(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return ['success' => false, 'errors' => $validator->errors()->all()];
        }

        $deviceToken = DeviceToken::where('token', $request->token)->first();

        if ($deviceToken) {
            return ['success' => true, 'message' => 'Already exists'];
        }

        $deviceToken          = new DeviceToken();
        $deviceToken->user_id = auth()->user()->id;
        $deviceToken->token   = $request->token;
        $deviceToken->is_app  = Status::NO;
        $deviceToken->save();

        return ['success' => true, 'message' => 'Token saved successfully'];
    }

    public function downloadAttachment($fileHash)
    {
        $filePath  = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $title     = slug(gs('site_name')) . '- attachments.' . $extension;
        try {
            $mimetype = mime_content_type($filePath);
        } catch (\Exception $e) {
            $notify[] = ['error', 'File does not exists'];
            return back()->withNotify($notify);
        }
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }

    public function purchasePackage(Request $request)
    {

        $request->validate([
            'id' => 'required|integer'
        ]);

        $package = Package::active()->findOrFail($request->id);
        $user    = auth()->user();

        if ($package->price > $user->balance) {
            $notify[] = ['error', 'Sorry, Insufficient balance'];
            return back()->withNotify($notify);
        }

        $user->package_id  = $package->id;
        $user->validity    = Carbon::now()->addDay($package->validity);
        $user->balance    -= $package->price;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $package->price;
        $transaction->post_balance = $user->balance;
        $transaction->charge       = 0;
        $transaction->trx_type     = '-';
        $transaction->details      = 'Purchased ' . $package->name;
        $transaction->trx          = getTrx();
        $transaction->remark       = 'purchase';
        $transaction->save();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = $user->username . ' has purchased ' . $package->name;
        $adminNotification->click_url = urlPath('admin.report.transaction', ['search' => $transaction->trx]);
        $adminNotification->save();

        notify($user, 'PURCHASE_COMPLETE', [
            'trx'              => $transaction->trx,
            'package'          => $package->name,
            'amount'           => showAmount($package->price, 2, currencyFormat: false),
            'post_balance'     => showAmount($user->balance, 2, currencyFormat: false),
            'validity'         => $package->validity . ' Days',
            'expired_validity' => showDateTime($user->validity),
            'purchased_at'     => showDateTime($transaction->created_at),
        ]);

        $notify[] = ['success', 'You have purchased ' . $package->name . ' successfully'];
        return to_route('user.transactions', ['search' => $transaction->trx])->withNotify($notify);
    }

    public function renewPackage(Request $request)
    {

        $request->validate([
            'id' => 'required|integer'
        ]);

        $package = Package::findOrFail($request->id);

        if (!$package->status) {
            $notify[] = ['info', 'Sorry, ' . $package->name . ' is not available to renew right now'];
            return to_route('user.home')->withNotify($notify);
        }

        $user = auth()->user();

        if ($user->package_id != $package->id) {
            $notify[] = ['error', 'Sorry, There is no package to renew'];
            return back()->withNotify($notify);
        }

        if ($package->price > $user->balance) {
            $notify[] = ['info', 'Sorry, Insufficient balance'];
            return back()->withNotify($notify);
        }

        $user->validity  = Carbon::parse($user->validity)->addDay($package->validity);
        $user->balance  -= $package->price;
        $user->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $package->price;
        $transaction->post_balance = $user->balance;
        $transaction->charge       = 0;
        $transaction->trx_type     = '-';
        $transaction->details      = 'Renewed ' . $package->name;
        $transaction->trx          = getTrx();
        $transaction->remark       = 'renew';
        $transaction->save();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = $user->username . ' has renewed ' . $package->name;
        $adminNotification->click_url = urlPath('admin.report.transaction', ['search' => $transaction->trx]);
        $adminNotification->save();

        notify($user, 'RENEW_COMPLETE', [
            'trx'              => $transaction->trx,
            'package'          => $package->name,
            'amount'           => showAmount($package->price, 2, currencyFormat: false),
            'post_balance'     => showAmount($user->balance, 2, currencyFormat: false),
            'validity'         => $package->validity . ' Days',
            'expired_validity' => showDateTime($user->validity),
            'renew_at'         => showDateTime($transaction->created_at),
        ]);

        $notify[] = ['success', 'You have renewed ' . $package->name . ' successfully'];
        return to_route('user.transactions', ['search' => $transaction->trx])->withNotify($notify);
    }

    public function signals(Request $request)
    {
        $pageTitle = 'Signals';
        $signals   = SignalHistory::where('user_id', auth()->user()->id);

        if ($request->search) {
            $signals = $signals->whereHas('signal', function ($signal) use ($request) {
                $signal->where('name', 'LIKE', '%' . $request->search . '%');
            });
        }

        $signals = $signals->orderBy('id', 'desc')->with('signal')->paginate(getPaginate());
        return view('Template::user.signals', compact('pageTitle', 'signals'));
    }

    public function referrals()
    {
        $user      = auth()->user();
        $pageTitle = 'Referrals';
        $maxLevel  = Referral::max('level');
        return view('Template::user.referrals', compact('pageTitle', 'user', 'maxLevel'));
    }
}
